const cron = require('node-cron');

const supabase = require('./supabase');
const { createApp } = require('./app');
const priceService = require('./services/priceService');
const { buildCronExpression, getPriceUpdateConfig } = require('./config/priceUpdates');

const app = createApp();
const PORT = process.env.PORT || 3001;
const priceUpdateConfig = getPriceUpdateConfig();
const cronExpression = buildCronExpression(priceUpdateConfig.time);

// Schedule daily price updates for local/server environments
cron.schedule(cronExpression, async () => {
    console.log(`Running scheduled price update for ${new Date().toISOString()}...`);
    try {
        await priceService.updateDailyPrices({ triggerSource: 'cron' });
        console.log('Daily price update completed successfully');
    } catch (error) {
        console.error('Daily price update failed:', error);
    }
}, {
    timezone: priceUpdateConfig.timezone,
});

// Initialize Supabase connection and start server
async function startServer() {
    try {
        // Test Supabase connection
        const connected = await supabase.testConnection();
        if (!connected) {
            throw new Error('Failed to connect to Supabase');
        }

        app.listen(PORT, () => {
            console.log(`🚀 ABOI Backend API running on port ${PORT}`);
            console.log(`📊 Health check: http://localhost:${PORT}/health`);
            console.log(`🌍 Environment: ${process.env.NODE_ENV || 'development'}`);
            console.log(`🗄️  Database: Supabase`);
            const allowedOrigins = Array.isArray(app.locals.allowedOrigins) && app.locals.allowedOrigins.length > 0
                ? app.locals.allowedOrigins.join(', ')
                : 'None (requests without Origin header only)';
            console.log(`🔐 Allowed origins: ${allowedOrigins}`);
            console.log(`⏰ Price update schedule: ${cronExpression} (${priceUpdateConfig.timezone})`);
        });
    } catch (error) {
        console.error('Failed to start server:', error);
        process.exit(1);
    }
}

// Graceful shutdown
process.on('SIGINT', async () => {
    console.log('\n🛑 Shutting down gracefully...');
    console.log('✅ Server shutdown complete');
    process.exit(0);
});

startServer();
