require('dotenv').config();
const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const morgan = require('morgan');

const authRoutes = require('./routes/auth');
const commodityRoutes = require('./routes/commodities-supabase');
const priceRoutes = require('./routes/prices');
const adminRoutes = require('./routes/admin');
const currencyRoutes = require('./routes/currency');

const isProduction = process.env.NODE_ENV === 'production';

const getDefaultOrigins = () => (
  isProduction
    ? ['https://your-domain.com', 'https://admin.your-domain.com']
    : ['http://localhost:5173', 'http://localhost:5174']
);

const parseEnvOrigins = (rawOrigins = '') => (
  rawOrigins
    .split(',')
    .map((origin) => origin.trim())
    .filter(Boolean)
);

function configureCors(app, allowedOrigins) {
  app.use(cors({
    origin: (origin, callback) => {
      if (!origin) {
        return callback(null, true);
      }

      if (allowedOrigins.includes(origin)) {
        return callback(null, true);
      }

      console.warn(`CORS blocked request from origin: ${origin}`);
      return callback(new Error('Not allowed by CORS'));
    },
    credentials: true,
  }));
}

function registerRoutes(app) {
  // Health check endpoint
  app.get('/health', (req, res) => {
    res.json({
      status: 'OK',
      timestamp: new Date().toISOString(),
      service: 'ABOI Backend API',
    });
  });

  // API Routes
  app.use('/api/auth', authRoutes);
  app.use('/api/commodities', commodityRoutes);
  app.use('/api/prices', priceRoutes);
  app.use('/api/admin', adminRoutes);
  app.use('/api/currency', currencyRoutes);
}

function registerErrorHandlers(app) {
  // Error handling middleware
  app.use((err, req, res, next) => {
    console.error('Error:', err);
    res.status(err.status || 500).json({
      error: {
        message: err.message || 'Internal Server Error',
        ...(process.env.NODE_ENV === 'development' && { stack: err.stack }),
      },
    });
  });

  // 404 handler
  app.use((req, res) => {
    res.status(404).json({
      error: {
        message: 'Route not found',
        path: req.originalUrl,
      },
    });
  });
}

function createApp() {
  const app = express();

  const defaultOrigins = getDefaultOrigins();
  const envOrigins = parseEnvOrigins(process.env.ALLOWED_ORIGINS);
  const allowedOrigins = envOrigins.length > 0 ? envOrigins : defaultOrigins;

  app.locals.allowedOrigins = allowedOrigins;

  app.use(helmet());
  configureCors(app, allowedOrigins);
  app.use(morgan('combined'));
  app.use(express.json());
  app.use(express.urlencoded({ extended: true }));

  registerRoutes(app);
  registerErrorHandlers(app);

  return app;
}

module.exports = {
  createApp,
};
